<?php

use Core\Shipment\Models\Order;
use Core\Shipment\Models\Shipment;
use Core\Contract\Models\Contract;
use Core\Contract\Models\ContractItem;
use Core\Shipment\Models\Scopes\ProviderEligibleShipmentsScope;
use Core\Financial\Models\Invoice;
use Carbon\Carbon;
use Core\Common\Models\Address;
use Core\Common\Models\Scopes\RelatedAddressesScope;
use Core\Provider\Models\TruckType;
use Core\Shipment\Models\ShipmentType;

if (!function_exists('contract_item')) {
    /**
     * get one contract item.
     * example 1: getting the contracted shipper price when making a new shipment.
     * example 2: getting the provider price when making interest on a shipment.
     *
     * @param  Shipment $shipment
     * @param  int      $contractable_id
     * @param  string   $contractable_type
     * @return ContractItem
     */
    function contract_item(Shipment $shipment, $contractable_id, $contractable_type = 'provider')
    {
        $contract = Contract::where('contractable_type', $contractable_type)
                            ->where('contractable_id', $contractable_id)
                            ->first();

        return $contract->items()
                        ->where('from_city_id', $shipment->from_city_id)
                        ->where('to_city_id', $shipment->to_city_id)
                        ->where('truck_type_id', $shipment->truck_type_id)
                        ->where('shipment_type_id', $shipment->shipment_type_id)
                        ->first();
    }
}

if (!function_exists('shipment_cost_average')) {
    /**
     * get the shipment average cost (mainly for the one-time shipper).
     *
     * @param  Shipment $shipment
     * @param  string   $contractable_type
     * @return double
     */
    function shipment_cost_average(Shipment $shipment, $contractable_type = 'shipper')
    {
        $prices = [];
        $items  = ContractItem::where('from_city_id', $shipment->from_city_id)
                              ->where('to_city_id', $shipment->to_city_id)
                              ->where('truck_type_id', $shipment->truck_type_id)
                              ->where('shipment_type_id', $shipment->shipment_type_id)
                              ->get();
        foreach ($items as $item) {
            if ($item->contract->contractable_type == $contractable_type) {
                $prices[] = $item->price;
            }
        }

        return collect($prices)->avg();
    }
}

if (!function_exists('shipment_eligible_providers')) {
    /**
     * all eligible providers for a shipment.
     *
     * @param  Shipment $shipment
     * @return array
     */
    function shipment_eligible_providers(Shipment $shipment)
    {
        
        $providers = [
            'interested' => collect(),
            'no_action'  => collect()
        ];

        $is_shipment_has_interests = $shipment->interests ? true : false;

        $items = ContractItem::where('from_city_id', $shipment->from_city_id)
                             ->where('to_city_id', $shipment->to_city_id)
                             ->where('truck_type_id', $shipment->truck_type_id)
                             ->where('shipment_type_id', $shipment->shipment_type_id)
                             ->get();
                             
                             
        foreach ($items as $item) {
            $contract = $item->contract;

// dd($item,$contract);
            if ($contract->contractable_type == 'provider') {
                $provider                    = $contract->contractable;
                $interest                    = $provider->interested($shipment->id);
                $provider->price             = $interest ? $interest->price : $item->price;
                $provider->margin            = number_format($shipment->cost - $provider->price, 2, '.', '');
                $provider->margin_percentage = number_format(($provider->margin / $shipment->cost) * 100, 2, '.', '');
                $provider->is_backhauling    = $shipment->id;

                if ($is_shipment_has_interests && $interest) {
                    $providers['interested']->push($provider);
                } else {
                    if ($shipment->provider_price && $shipment->provider_id == $provider->id) {
                        $provider->price = $shipment->provider_price;
                    }

                    $providers['no_action']->push($provider);
                }
            }
        }
        $providers['interested'] = $providers['interested']->sortByDesc('margin')->sortByDesc('is_backhauling');
        $providers['no_action']  = $providers['no_action']->sortByDesc('margin')->sortByDesc('is_backhauling');

        return $providers;
    }
}

if (!function_exists('provider_eligible_shipments')) {
    /**
     * all eligible shipments for a provider.
     *
     * @param  int $provider_id
     * @return array
     */
    function provider_eligible_shipments($provider_id)
    {
        $array    = [];
        $contract = Contract::where('contractable_type', 'provider')
                            ->where('contractable_id', $provider_id)
                            ->first();

        foreach ($contract->items as $item) {
            $shipments = Shipment::withoutGlobalScope(ProviderEligibleShipmentsScope::class)
                                 ->where('status', config('core_shipment.statuses.shipment.ready.value'))
                                 ->where('from_city_id', $item->from_city_id)
                                 ->where('to_city_id', $item->to_city_id)
                                 ->where('truck_type_id', $item->truck_type_id)
                                 ->where('shipment_type_id', $item->shipment_type_id)
                                 ->with('interests')
                                 ->get();

            if ($shipments) {
                foreach ($shipments as $shipment) {
                    if (!in_array($provider_id, $shipment->interests->pluck('provider_id')->toArray())) {
                        $array[] = $shipment->toArray();
                    }
                }
            }
        }

        return $array ? collect($array) : collect();
    }
}

if (!function_exists('handle_invoice')) {
    /**
     * handle invoice
     *
     * @param  Order  $order
     * @param  string $currency
     * @return Invoice
     */
    function handle_invoice(Order $order, $currency = 'SAR')
    {
        if(!$order->invoice) {
            if ($order->shipper->contract || request()->payment_method) {
                $items                 = [];
                $total                 = 0.00;
                $is_contracted_shipper = $order->shipper->contract ? true : false;
                $add_item              = $is_contracted_shipper ? true : false;

                foreach ($order->shipments as $shipment) {
                    if (!$add_item) {
                        $add_item = $shipment->status == config('core_shipment.statuses.shipment.confirmed.value') ?
                                    true :
                                    false;
                    }

                    if ($add_item) {
                        $total += $shipment->cost;
                        $items[] = $shipment->invoiceItem()->make(['price' => $shipment->cost])->toArray();
                        
                        // reseting to false (as a one-time shipper I may pay for 3 shipments of 5 shipments)
                        if (!$is_contracted_shipper) {
                            $add_item = false;
                        }
                    }
                }

                $invoice = $order->invoice()->create([
                    'status'   => config('core_financial.statuses.invoice.pending.value'),
                    'total'    => $total,
                    'due_date' => $order->pickup_date,
                    'currency' => $currency
                ]);

                $invoice->items()->createMany($items);

                if (isset(request()->attachments)) {
                    $current_attachments = [];
                    $label               = request()->payment_method;
                    
                    foreach (request()->attachments as $file) {
                        $current_attachments[] = \Core\Common\Models\Attachment::make([
                            'label'   => $label,
                            'path'    => $file,
                            'details' => ['label' => $label]
                        ])->toArray();
                    }

                    $invoice->attachments()->createMany($current_attachments);
                }

                return $invoice;
            }
        }

        return $order->invoice;
    }
}

if (!function_exists('compare_times')) {
    /**
     * compare between 2 times.
     *
     * @param  mixed  $first_time
     * @param  string $operator
     * @param  mixed  $second_time
     * @return bool
     */
    function compare_times($first_time, $operator, $second_time)
    {
        return Carbon::parse($first_time)->{$operator}(Carbon::parse($second_time));
    }

    if (!function_exists('truck_shipment_types')) {
        /**
         * get the appropriate truck types and shipment types.
         *
         * @return array
         */
        function truck_shipment_types()
        {
            $data    = []; 
            $shipper = null;

            if (auth()->user()->userable_type == 'shipper') {
                $shipper = auth()->user()->userable;
            } elseif (request()->shipper_id) {
                $shipper = \Core\Shipper\Models\Shipper::find(request()->shipper_id);
            }

            if($shipper->contract) {
                $already_added_trucks = [];
                $from_city_id = Address::withoutGlobalScope(RelatedAddressesScope::class)
                                       ->find(request()->from_address_id)->city_id;
                $to_city_id   = Address::withoutGlobalScope(RelatedAddressesScope::class)
                                       ->find(request()->to_address_id)->city_id;

                $items = $shipper->contract->items()->where('from_city_id', $from_city_id)
                                                    ->where('to_city_id', $to_city_id)
                                                    ->get();

                foreach($items as $item) {
                    $truck_type = $item->truckType;

                    if (isset($already_added_trucks[(string) $truck_type->id])) {
                        $data[$already_added_trucks[(string) $truck_type->id]]->shipment_types->push($item->shipmentType);
                    } else {
                        $truck_type->shipment_types = collect();
                        $truck_type->shipment_types->push($item->shipmentType);
                        $data[] = $truck_type;
                        $already_added_trucks[(string) $truck_type->id] = sizeof($data) - 1;
                    }
                }

                return $data;
            }

            $truck_types    = TruckType::all();
            $shipment_types = ShipmentType::all();
            
            foreach ($truck_types as $truck_type) {
                $truck_type->shipment_types = $shipment_types;
                $data[] = $truck_type;
            }

            return $data;
        }
    }
}
