<?php

namespace Core\Shipment\Tests\Feature;

use Core\Base\Tests\TestCase;
use Core\Shipment\Models\Order as Model;
use Core\Shipment\Models\Shipment;
use Core\Provider\Models\Provider;
use Core\Shipper\Models\Shipper;
use Core\Auth\Models\User;

class OrderTest extends TestCase
{
    /**
     * the base url
     *
     * @var string
     */
    protected $base_url;

    /**
     * the data that will be sent in the request (create/update)
     *
     * @var array
     */
    protected $data;

    /**
     * the json response
     *
     * @var array
     */
    protected $json;

    /**
     * the current model
     *
     * @var Model
     */
    protected $model;

    /**
     * setting up
     *
     * @throws \ReflectionException
     * @return void
     */
    public function setUp(): void
    {
        parent::setUp();

        $this->base_url     = $this->getApiBaseUrl() . 'orders/';
        $this->model        = Model::factory()->make();
        $this->data         = $this->model->toArray();
        $this->json         = $this->getJsonStructure();
        $this->json['data'] = ['id'];
        $removed_attributes = ['shipper_id', 'from_address_id', 'to_address_id'];

        foreach ($this->data as $key => $value) {

            if(in_array($key, $removed_attributes)) {
                continue;
            }

            $this->json['data'][] = $key;
        }
    }

    /**
     * create new entry
     *
     * @return Model
     */
    protected function getNewEntry()
    {
        $custom_attributes = ['order_id' => null, 'truck_type_qty' => 3];
        $this->data['shipments'] = Shipment::factory()->count(3)->make($custom_attributes)->toArray();
        return Model::factory()->create();
    }

    /**
     * get the id
     *
     * @return int
     */
    protected function getId()
    {
        return $this->getNewEntry()->id;
    }

    /**
     * handle shipper
     */
    public function handleShipper($order, $shipments, $fcm_token, $create_contract = true) {
        $shipper = $order->shipper;

        // creating a user
        $user = User::factory()->make();
        $user = $user->toArray();
        $user['password'] = '123456';
        $user['fcm_tokens'] = $fcm_token;
        $shipper->user()->create($user);

        // contract details
        if ($create_contract) {
            $contract = $shipper->contract ?? $shipper->contract()->create();
            $from_city_id = $order->fromAddress->city_id;
            $to_city_id = $order->toAddress->city_id;
    
            foreach ($shipments as $shipment) {
                $contract->items()->create([
                    'from_city_id'     => $from_city_id,
                    'to_city_id'       => $to_city_id,
                    'truck_type_id'    => $shipment->truck_type_id,
                    'shipment_type_id' => $shipment->shipment_type_id,
                    'price'            => 100.00
                ]);
            }
        }
    }

    /**
     * handle providers
     */
    public function handleProviders($order, $shipment, $fcm_token) {
        $partner = Provider::factory()->create();
        $user = User::factory()->make();
        $user = $user->toArray();
        $user['password'] = '123456';
        $user['fcm_tokens'] = $fcm_token;
        $partner->user()->create($user);
        $contract = $partner->contract()->create();
        $from_city_id = $order->fromAddress->city_id;
        $to_city_id = $order->toAddress->city_id;

        $contract->items()->create([
            'from_city_id'     => $from_city_id,
            'to_city_id'       => $to_city_id,
            'truck_type_id'    => $shipment->truck_type_id,
            'shipment_type_id' => $shipment->shipment_type_id,
            'price'            => 100.00
        ]);
    }

    /**
     * Display a listing of the resource.
     *
     * @return void
     */
    public function testItShouldGetListingOfTheResource()
    {
        $this->getNewEntry();
        $current_json              = $this->json;
        $current_json['data']      = [];
        $current_json['data']['*'] = $this->json['data'];

        $this->json('GET', $this->base_url, [], $this->getHeaders())
             ->assertStatus(200)
             ->assertJsonStructure($current_json);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @return void
     */
    public function testItShouldStoreNewlyCreatedResource()
    {
        $custom_attributes = ['order_id' => null, 'truck_type_qty' => 3];
        
        $shipments = Shipment::factory()->count(3)->make($custom_attributes);
        $this->data['shipments'] = $shipments->toArray();

        $fcm_token = '';
        $this->handleShipper($this->model, $shipments, $fcm_token, false);

        $this->json('POST', $this->base_url, $this->data, $this->getHeaders())
             ->assertStatus(201)
             ->assertJsonStructure($this->json);
    }

    /**
     * Store a newly created resource in storage (Ready and has providers that meet the criteria).
     *
     * @return void
     */
    public function testItShouldStoreNewlyCreatedResourceThatHasReadyShipmentsAndProvidersThatMeetTheCriteria()
    {
        $custom_attributes = [
            'order_id'       => null,
            'truck_type_qty' => 3,
            'from_city_id'   => $this->model->fromAddress->city_id,
            'to_city_id'     => $this->model->toAddress->city_id
        ];

        $shipments = Shipment::factory()->count(3)->make($custom_attributes);
        $this->data['shipments'] = $shipments->toArray();

        $fcm_token = '';
        $this->handleShipper($this->model, $shipments, $fcm_token);

        $fcm_token = '';
        $this->handleProviders($this->model, $shipments[0], $fcm_token);

        $fcm_token = '';
        $this->handleProviders($this->model, $shipments[0], $fcm_token);

        $fcm_token = '';
        $this->handleProviders($this->model, $shipments[1], $fcm_token);

        $this->json('POST', $this->base_url, $this->data, $this->getHeaders())
             ->assertStatus(201)
             ->assertJsonStructure($this->json);
    }

    /**
     * Store a newly created resource in storage (Pending shipments, this is for one-time shipper).
     *
     * @return void
     */
    public function testItShouldStoreNewlyCreatedResourceThatHasPendingShipmentsThatHaveBeenCreatedByOneTimeShipper()
    {
        $custom_attributes = [
            'order_id'       => null,
            'truck_type_qty' => 3,
            'from_city_id'   => $this->model->fromAddress->city_id,
            'to_city_id'     => $this->model->toAddress->city_id
        ];
        
        $shipments = Shipment::factory()->count(3)->make($custom_attributes);
        $this->data['shipments'] = $shipments->toArray();

        $fcm_token = '';
        $this->handleShipper($this->model, $shipments, $fcm_token);

        $fcm_token = '';
        $one_time_shipper = Shipper::factory()->create();
        $user = User::factory()->make();
        $user = $user->toArray();
        $user['password'] = '123456';
        $user['fcm_tokens'] = $fcm_token;
        $one_time_shipper->user()->create($user);

        $this->data['shipper_id'] = $one_time_shipper->id;

        $this->json('POST', $this->base_url, $this->data, $this->getHeaders())
             ->assertStatus(201)
             ->assertJsonStructure($this->json);
    }

    /**
     * Display the specified resource.
     *
     * @return void
     */
    public function testItShouldGetSpecifiedResource()
    {
        $this->json('GET', $this->base_url . $this->getId(), [], $this->getHeaders())
             ->assertStatus(200)
             ->assertJsonStructure($this->json);
    }

    /**
     * update a resource in storage.
     *
     * @return void
     */
    public function testItShouldUpdateSpecifiedResource()
    {
        $this->json('PUT', $this->base_url . $this->getId(), $this->data, $this->getHeaders())
             ->assertStatus(200)
             ->assertJsonStructure($this->json);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @return void
     */
    public function testItShouldRemoveSpecifiedResource()
    {
        $this->json['data'] = [];
        $this->json('DELETE', $this->base_url . $this->getId(), [], $this->getHeaders())
             ->assertStatus(200)
             ->assertJsonStructure($this->json);
    }
}
