<?php

namespace Core\Shipment\Observers;

use Core\Shipment\Models\Shipment;
use Core\Shipment\Jobs\ProcessShipmentChange;
use Illuminate\Support\Str;

class ShipmentObserver
{
    /**
     * Handle the Shipment "saving" event.
     *
     * @param  \Core\Shipment\Models\Shipment  $shipment
     * @return void
     */
    public function saving(Shipment $shipment)
    {
        // new shipment
        if (!$shipment->created_at) {
            $shipment->status            = $this->statuses()['pending']['value'];
            $shipment->from_city_id      = $shipment->order->fromAddress->city_id;
            $shipment->to_city_id        = $shipment->order->toAddress->city_id;
            $shipment->pickup_date       = $shipment->order->pickup_date;
            $shipment->pickup_from_time  = $shipment->order->pickup_from_time;
            $shipment->pickup_to_time    = $shipment->order->pickup_to_time;
            $shipment->dropoff_from_time = $shipment->order->dropoff_from_time;
            $shipment->dropoff_to_time   = $shipment->order->dropoff_to_time;
            $shipment->cost              = 100; // this is just for calling the set method
            $shipper                     = $shipment->order->shipper;

            if ($shipper->contract) {
                $shipment->status = $this->statuses()['ready']['value'];
            }
        }

        // back and forth changes for the one-time shipper like: admin updated cost
        if (request()->status == $this->statuses()['approved']['value'] || request()->cost) {
            $shipment->status = $this->statuses()['reviewed']['value'];
        }
        if (request()->modification_type) {
            $modification_type = request()->modification_type;
            $shipment->status = $this->statuses()['modification-request']['value'];

            if ($modification_type != $this->modifications()['other']['value']) {
                $shipment->modification_reason = __("shipment::modification.{$modification_type}");
            }
        }

        // assigned provider
        if ($shipment->provider_id && !$shipment->driver_id) {
            $shipment->provider_price = 100; // this is just for calling the set method
            $shipment->status         = $this->statuses()['partner-assigned']['value'];

            if ($shipment->provider->type == 'freelancer') {
                $shipment->driver_id = $shipment->provider->driver->id;
            }
        }

        // assigned driver
        if ($shipment->provider_id && $shipment->driver_id) {
            $shipment->status = $this->statuses()['driver-assigned']['value'];
        }

        // status already sent in the request
        if (in_array($this->routeName(), ['api.v1.shipments.store', 'api.v1.shipments.update']) &&
            request()->status && 
            $shipment->status != $this->statuses()['reviewed']['value'] &&
            $shipment->status != $this->statuses()['modification-request']['value']) {
            $shipment->status = request()->status;
        }
    }

    /**
     * Handle the Shipment "saved" event.
     *
     * @param  \Core\Shipment\Models\Shipment  $shipment
     * @return void
     */
    public function saved(Shipment $shipment)
    {
        $allow_process_shipment_change = $shipment->status == $this->statuses()['pending']['value'] ? false : true;

        if (request()->latitude && request()->longitude) {
            $allow_process_shipment_change = false;
        }

        // attachments that will be uploaded by the driver
        if (isset(request()->attachments)) {
            $auth_user = auth()->user();
            if ($auth_user && ($auth_user->userable_type == 'provider' || $auth_user->userable_type == 'driver')) {
                // $label = $this->statuses()[Str::lower(request()->status)]['localization'];
                // $label = request()->labels;
                foreach (request()->attachments as $key =>$file) {
                    $shipment->attachments()->create([
                        'label'   => request()->labels[$key],
                        'path'    => $file,
                        'details' => ['label' => request()->labels[$key]]
                    ]);
                }
            }
        }

        if(in_array($this->requestMethod(), ['POST', 'PUT', 'PATCH']) && $allow_process_shipment_change) {
            ProcessShipmentChange::dispatch($shipment, auth()->user());
        }
    }

    /**
     * Get the current route name.
     *
     * @return string
     */
    protected function routeName()
    {
        return request()->route() ? request()->route()->getName() : '';
    }

    /**
     * Get the current request method.
     *
     * @return string
     */
    protected function requestMethod()
    {
        return request()->getMethod();
    }

    /**
     * shipment statuses config
     *
     * @return array
     */
    protected function statuses()
    {
        return config('core_shipment.statuses.shipment');
    }

    /**
     * shipment modifications config
     *
     * @return array
     */
    protected function modifications()
    {
        return config('core_shipment.modifications.shipment');
    }
}
