<?php

namespace Core\Shipment\Models;

use Carbon\Carbon;
use Core\Base\Models\Base;
use Illuminate\Database\Eloquent\Casts\Attribute;
use Core\Common\Resources\AttachmentResource;

class Shipment extends Base
{
    /**
     * The "booted" method of the model.
     *
     * @return void
     */
    protected static function booted()
    {
        static::addGlobalScope(new Scopes\RelatedShipmentsScope);
        static::addGlobalScope(new Scopes\StatusScope);
        static::addGlobalScope(new Scopes\ProviderEligibleShipmentsScope);
        static::addGlobalScope(new Scopes\OrderByLatestScope);
    }

    /**
     * get the order.
     */
    public function order()
    {
        return $this->belongsTo(Order::class, 'order_id');
    }

    /**
     * get the provider.
     */
    public function provider()
    {
        return $this->belongsTo(\Core\Provider\Models\Provider::class, 'provider_id');
    }

    /**
     * get the driver.
     */
    public function driver()
    {
        return $this->belongsTo(\Core\Provider\Models\Driver::class, 'driver_id');
    }

    /**
     * get from city.
     */
    public function fromCity()
    {
        return $this->belongsTo(\Core\Common\Models\City::class, 'from_city_id');
    }

    /**
     * get to city.
     */
    public function toCity()
    {
        return $this->belongsTo(\Core\Common\Models\City::class, 'to_city_id');
    }

    /**
     * get the truck type.
     */
    public function truckType()
    {
        return $this->belongsTo(\Core\Provider\Models\TruckType::class, 'truck_type_id');
    }

    /**
     * get the shipment type.
     */
    public function shipmentType()
    {
        return $this->belongsTo(ShipmentType::class, 'shipment_type_id');
    }

    /**
     * get the commodity.
     */
    public function commodity()
    {
        return $this->belongsTo(Commodity::class, 'commodity_id');
    }

    /**
     * get the UOM.
     */
    public function uom()
    {
        return $this->belongsTo(UOM::class, 'uom_id');
    }

    /**
     * get the interests.
     */
    public function interests()
    {
        return $this->hasMany(Interest::class, 'shipment_id');
    }

    /**
     * get the backhauls.
     */
    public function backhauls()
    {
        return $this->hasMany(Backhaul::class, 'shipment_id');
    }

    /**
     * set the cost.
     * 
     * @param  double $value
     * @return void
     */
    public function setCostAttribute($value)
    {
        if($this->order->shipper->contract) {
            $cost = contract_item($this, $this->order->shipper_id, 'shipper')->price;
        } else {
            if (request()->cost) {
                $cost = request()->cost;
            } else {
                $cost = shipment_cost_average($this);
            }
        }

        $this->attributes['cost'] = $cost;
    }

    /**
     * set the provider price.
     * 
     * @param  double $value
     * @return void
     */
    public function setProviderPriceAttribute($value)
    {
        $interest = $this->provider->interested($this->id);
        
        if ($interest) {
            $price = ($interest)->price;
        } else {
            $price = contract_item($this, $this->provider_id)->price;
        }
        
        // to change the price for the eligible service providers
        if(request()->price) {
            $price = request()->price;
            
            if ($interest) {
                ($interest)->update(['price' => $price]);
            }
        }

        $this->attributes['provider_price'] = $price;
    }

    /**
     * get the provider price.
     * 
     * @param  double $value
     * @return void
     */
    public function getProviderPriceAttribute($value)
    {
        if (auth()->user() && 
            auth()->user()->userable_type == 'provider' &&
            !$value &&
            $this->status == config('core_shipment.statuses.shipment.ready.value')) {

            /**
             * @todo cache the data as this query takes a lot of time
             * (we can do that when creating/updating the contract items) and the code 
             * below is wokring only on one route (shipment details) for better performance
             * it should be working on all routes
             */
            if (request()->shipment) {
            $item = auth()->user()->userable->contract->items()->where('from_city_id', $this->from_city_id)
                                                               ->where('to_city_id', $this->to_city_id)
                                                               ->where('truck_type_id', $this->truck_type_id)
                                                               ->where('shipment_type_id', $this->shipment_type_id)
                                                               ->first();
            $value = $item ? $item->price : null;
            }
        }

        return $value;
    }

    /**
     * allow/refuse the driver to start the trip
     *
     * @return bool
     */
    public function getAllowDriverToStartAttribute()
    {
        $carbon = Carbon::parse($this->pickup_date . ' ' . $this->pickup_from_time);
        $diff   = Carbon::now()->diffInHours($carbon);
        return $diff <= 24;
    }

    /**
     * a flag to show wether the shipment needs an urgent action or not
     *
     * @return bool
     */
    public function getUrgentActionNeededAttribute() : bool
    {
        $alerting_key = "shipment-{$this->id}-alert";

        if (cache()->has($alerting_key)) {
            $alerting_item = cache()->get($alerting_key);
            $array = [
                $this->statuses()['driver-to-pickup-address']['value'],
                $this->statuses()['driver-to-dropoff-address']['value']
            ];

            if (in_array($this->status, $array) && compare_times($alerting_item['expires_at'], 'lte', now())) {
                return true;
            }
        }
        
        if (!$this->driver_id) {
            return compare_times($this->pickup_date . ' ' . $this->pickup_from_time, 'lte', now());
        }

        return false;
    }

    /**
     * get the attachments.
     */
    public function attachments()
    {
        return $this->morphMany(\Core\Common\Models\Attachment::class, 'attachmentable');
    }

    /**
     * get sorted attachments.
     */
    public function getSortedAttachmentsAttribute()
    {
        $data = [];

        foreach ($this->attachments->groupBy('label') as $key => $attachments_group) {
            $data[$key] = AttachmentResource::collection($attachments_group);
        }

        return $data;
    }

    /**
     * get the invoice item.
     */
    public function invoiceItem()
    {
        return $this->morphOne(\Core\Financial\Models\InvoiceItem::class, 'itemable');
    }

    /**
     * get the requests.
     */
    public function requests()
    {
        return $this->morphMany(\Core\Financial\Models\Request::class, 'requestable');
    }

    /**
     * Interact with the status (I18n).
     *
     * @return \Illuminate\Database\Eloquent\Casts\Attribute
     */
    public function statusI18n(): Attribute
    {
        return new Attribute(
            get: fn () => __("shipment::status.{$this->status}")
        );
    }

    /**
     * shipment statuses config
     *
     * @return array
     */
    protected function statuses()
    {
        return config('core_shipment.statuses.shipment');
    }

    /**
     * Interact with the code.
     *
     * @return \Illuminate\Database\Eloquent\Casts\Attribute
     */
    public function code(): Attribute
    {
        return new Attribute(
            get: fn () => "ELD{$this->id}"
        );
    }
}
