<?php

namespace Core\Shipment\Jobs;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldBeUnique;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Core\Shipment\Models\Shipment;
use Core\Auth\Models\User;
use Illuminate\Support\Facades\Notification;
use Core\Shipment\Notifications\BackAndForthShipmentChanges;
use Core\Shipment\Notifications\ReadyShipment;
use Core\Shipment\Notifications\ShipmentPartnerAssigned;
use Core\Shipment\Notifications\ShipmentDriverAssigned;
use Core\Shipment\Notifications\ShipmentDriverUpdatedStatus;
use Core\Shipment\Notifications\CanceledShipment;
use Illuminate\Support\Str;
use Core\Shipment\Jobs\HandleShipmentAlert;

class ProcessShipmentChange implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    /**
     * shipment
     * 
     * @var Shipment
     */
    protected $shipment;

    /**
     * auth user
     * 
     * @var User
     */
    protected $auth_user;

    /**
     * Create a new job instance.
     *
     * @param  Shipment $shipment
     * @param  User     $auth_user
     * @return void
     */
    public function __construct(Shipment $shipment, User $auth_user)
    {
        $this->shipment  = $shipment;
        $this->auth_user = $auth_user;
    }

    /**
     * Execute the job.
     *
     * @return void
     */
    public function handle()
    {
        if ($this->shipment->status == $this->statuses()['ready']['value']) {
            $providers = shipment_eligible_providers($this->shipment)['no_action'];

            if($this->auth_user->userable_type == 'shipper') {
                $admins = \Core\Auth\Models\User::where('userable_type', null)->get();
                Notification::send($admins, new ReadyShipment($this->shipment));
            }

            foreach ($providers as $provider) {
                $provider->user->notify(new ReadyShipment($this->shipment, $provider->price));
            }

            return;
        }

        if ($this->shipment->status == $this->statuses()['partner-assigned']['value']) {
            $users = collect([$this->shipment->order->shipper->user, $this->shipment->provider->user]);
            Notification::send($users, new ShipmentPartnerAssigned($this->shipment));

            return;
        }

        if ($this->shipment->status == $this->statuses()['driver-assigned']['value']) {
            $users        = null;
            $shipper_user = $this->shipment->order->shipper->user;
            
            if ($this->shipment->provider->type == 'partner') {
                $users = \Core\Auth\Models\User::where('userable_type', null)->get();
                $users->push($this->shipment->driver->user);
            } else {
                $users = collect([$this->shipment->provider->user]);
            }

            $users->push($shipper_user);

            Notification::send($users, new ShipmentDriverAssigned($this->shipment));

            return;
        }

        // back and forth changes for the one-time shipper and the admin
        if ($this->shipment->status == $this->statuses()['reviewed']['value']){
            $this->shipment->order->shipper->user->notify(new BackAndForthShipmentChanges($this->shipment));
            return;
        }
        if ($this->shipment->status == $this->statuses()['modification-request']['value']) {
            $admins = \Core\Auth\Models\User::where('userable_type', null)->get();
            $title  = 'Modification Request';
            $body   = 'The shipper requested modification';
            Notification::send($admins, new BackAndForthShipmentChanges($this->shipment, $title, $body));
            return;
        }
        if ($this->shipment->status == $this->statuses()['confirmed']['value']) {
            $admins = \Core\Auth\Models\User::where('userable_type', null)->get();
            $title  = 'Shipment Confirmed!';
            $body   = 'The shipment has been confirmed.';
            Notification::send($admins, new BackAndForthShipmentChanges($this->shipment, $title, $body));
            return;
        }
        if($this->shipment->status == $this->statuses()['canceled']['value']) {
            $users = collect([$this->shipment->order->shipper->user]);
            if($this->auth_user->userable_type == 'shipper') {
                $users = \Core\Auth\Models\User::where('userable_type', null)->get();
            }

            Notification::send($users, new CanceledShipment($this->shipment));
            return;
        }

        // driver updates the status
        $admins = \Core\Auth\Models\User::where('userable_type', null)->get();
        $users  = $admins->push($this->shipment->order->shipper->user);

        if ($this->shipment->provider->type == 'partner') {
            $users->push($this->shipment->provider->user);
        }

        $localization = $this->statuses()[Str::lower($this->shipment->status)]['localization'];
        Notification::send($users, new ShipmentDriverUpdatedStatus($this->shipment, $localization));

        // Alerting (ex: late shipments)
        $alerting_array = [
            $this->statuses()['driver-to-pickup-address']['value'],
            $this->statuses()['driver-to-dropoff-address']['value']
        ];

        $alerting_key = "shipment-{$this->shipment->id}-alert";
        if (in_array($this->shipment->status, $alerting_array) && $this->shipment->eta) {
            $eta_minutes = $this->shipment->eta;
            $minutes = $eta_minutes + (config('core_shipment.alerting.shipment.late-percentage') * $eta_minutes) / 100;
            $time = now()->addMinutes($minutes);
            
            cache()->put($alerting_key, ['expires_at' => $time], now()->addMinutes($minutes)->addHours(3));
            HandleShipmentAlert::dispatch($this->shipment)->delay($time);
        }

        if($this->shipment->status == $this->statuses()['delivered']['value']) {
            $this->shipment->requests()->create([
                'category_id'          => 1, // for now static, we have to get/create Deposits category
                'beneficiaryable_type' => 'provider',
                'beneficiaryable_id'   => $this->shipment->provider_id,
                'status'               => config('core_financial.statuses.request.pending.value'),
                'total'                => $this->shipment->provider_price
            ]);

            // Backhauling
            $admins                = \Core\Auth\Models\User::where('userable_type', null)->get();
            $provider              = $this->shipment->provider;
            $backhauling_shipments = Shipment::where('status', $this->statuses()['ready']['value'])
                                             ->where('from_city_id', $this->shipment->to_city_id)
                                             ->where('to_city_id', $this->shipment->from_city_id)
                                             ->where('truck_type_id', $this->shipment->truck_type_id)
                                             ->where('shipment_type_id', $this->shipment->shipment_type_id)
                                             ->get();
            
            foreach ($backhauling_shipments as $backhauling_shipment) {
                $backhauling_shipment->backhauls()->create([
                    'shipment_id' => $backhauling_shipment->id,
                    'provider_id' => $provider->id
                ]);
                Notification::send($admins, new ReadyShipment($backhauling_shipment, $provider->price, true));
                $provider->user->notify(new ReadyShipment($backhauling_shipment, $provider->price, true));
            }

            // promotions
            handle_promotions_trackers($provider->user, $this->shipment);
        }
    }

    /**
     * shipment statuses config
     *
     * @return array
     */
    protected function statuses()
    {
        return config('core_shipment.statuses.shipment');
    }
}
