<?php

namespace Core\Shipment\Jobs;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldBeUnique;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Core\Shipment\Models\Shipment;
use Illuminate\Support\Facades\Notification;
use Core\Shipment\Notifications\ShipmentAlert;

class HandleShipmentAlert implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    /**
     * shipment
     * 
     * @var Shipment
     */
    protected $shipment;

    /**
     * Create a new job instance.
     *
     * @param  Shipment $shipment
     * @return void
     */
    public function __construct(Shipment $shipment)
    {
        $this->shipment = $shipment;
    }

    /**
     * Execute the job.
     *
     * @return void
     */
    public function handle()
    {
        $alerting_array = [
            $this->statuses()['driver-to-pickup-address']['value'],
            $this->statuses()['driver-to-dropoff-address']['value']
        ];

        $alerting_key   = "shipment-{$this->shipment->id}-alert";
        $alerting_item  = cache()->get($alerting_key);
        $current_status = $this->shipment->status;
        
        if (in_array($current_status, $alerting_array) && compare_times($alerting_item['expires_at'], 'lte', now())) {
            $admins = \Core\Auth\Models\User::where('userable_type', null)->get();
            Notification::send($admins, new ShipmentAlert($this->shipment));
        }
    }

    /**
     * shipment statuses config
     *
     * @return array
     */
    protected function statuses()
    {
        return config('core_shipment.statuses.shipment');
    }
}
