<?php

namespace Core\Shipment\Controllers\API\V1;

use Core\Shipment\Requests\OrderRequest as FormRequest;
use Core\Shipment\Models\Order as Model;
use Core\Shipment\Resources\OrderResource as Resource;
use Core\Financial\Jobs\HandleNewPendingInvoice;

class OrderController extends \Core\Base\Controllers\API\Controller
{
    /**
     * Init.
     * @param FormRequest $request
     * @param Model       $model
     * @param string      $resource
     */
    public function __construct(FormRequest $request, Model $model, $resource = Resource::class)
    {
        parent::__construct($request, $model, $resource);
    }
    
    /**
     * Pay the invoice.
     *
     * @param  int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function pay($id)
    {
        $order   = $this->model->find($id);
        $invoice = handle_invoice($order);

        if (request()->payment_method == config('core_financial.payment.methods.bank-transfer.value')) {
            HandleNewPendingInvoice::dispatch($invoice, auth()->user());
            return $this->sendResponse([], 'Your invoice will be reviewed shortly');
        }

        if (request()->payment_method == config('core_financial.payment.methods.online.value')) {
            $payload = [
                'trackid'       => env('APP_ENV') . $invoice->id, // prefix here is just for testing and will be removed when I'm done from testing
                'customerEmail' => $order->shipper->user->email,
                'merchantIp'    => request()->ip(),
                'country'       => 'SA', // needs to be dynamic
                'currency'      => $invoice->currency,
                'amount'        => $invoice->total
            ];

            $urway    = new \Core\Financial\PaymentGateways\Urway($payload);
            $purchase = $urway->purchase();

            if ($purchase) {
                return $this->sendResponse($purchase);
            }

            return $this->sendResponse([], 'Error.', false, 500);
        }
    }

    /**
     * check if the transaction (invoice) for the order is successful or not.
     *
     * @param  int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function check($id)
    {
        $order   = $this->model->find($id);
        $invoice = $order->invoice;
        $payload = [
            'transid'    => request()->TranId,
            'trackid'    => env('APP_ENV') . $invoice->id, // prefix here is just for testing and will be removed when I'm done from testing
            'merchantIp' => request()->ip(),
            'currency'   => $invoice->currency,
            'amount'     => $invoice->total
        ];

        $urway               = new \Core\Financial\PaymentGateways\Urway($payload);
        $transaction_inquiry = $urway->transactionInquiry();

        if ($transaction_inquiry['responseCode'] == '000') {
            $invoice->update(['status' => config('core_financial.statuses.invoice.paid.value')]);
            return $this->sendResponse();
        }

        return $this->sendResponse([], 'Error.', false, 500);
    }

    /**
     * prepare the order
     *
     * @param  int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function prepare()
    {
        // pickup & dropoff addressess (scopes contains the logic)
        if ((auth()->user()->userable_type == 'shipper' && sizeof(request()->all()) == 0) || 
            (request()->shipper_id && sizeof(request()->all()) == 1) || 
            (request()->from_address_id && !request()->to_address_id)) {
            return \Core\Shipper\Resources\GroupResource::collection(\Core\Shipper\Models\Group::all());
        }
        
        // truck types, shipment types, commodities and UOMs 
        if ((auth()->user()->userable_type == 'shipper' || request()->shipper_id) && 
            request()->from_address_id && request()->to_address_id) {
            return $this->sendResponse([
                'truck_types' => truck_shipment_types(),
                'commodities' => \Core\Shipment\Resources\CommodityResource::collection(\Core\Shipment\Models\Commodity::all()),
                'uom'         => \Core\Shipment\Resources\UOMResource::collection(\Core\Shipment\Models\UOM::all()),
            ]);
        }

        return $this->sendResponse([], 'Error.', false, 422);
    }
}
