<?php

namespace Core\Provider\Tests\Feature;

use Core\Base\Tests\TestCase;
use Core\Provider\Models\Provider as Model;
use Core\Auth\Models\User;
use Core\Provider\Models\Driver;
use Core\Provider\Models\Truck;

class ProviderTest extends TestCase
{
    /**
     * the base url
     *
     * @var string
     */
    protected $base_url;

    /**
     * the data that will be sent in the request (create/update)
     *
     * @var array
     */
    protected $data;

    /**
     * the json response
     *
     * @var array
     */
    protected $json;

    /**
     * setting up
     *
     * @throws \ReflectionException
     * @return void
     */
    public function setUp(): void
    {
        parent::setUp();
        $this->base_url     = $this->getApiBaseUrl() . 'providers/';
        $this->data         = Model::factory()->make(['type' => 'freelancer'])->toArray();
        $this->json         = $this->getJsonStructure();
        $this->json['data'] = ['id'];

        foreach ($this->data as $key => $value) {
            $this->json['data'][] = $key;
        }
    }

    /**
     * create new entry
     *
     * @return Model
     */
    protected function getNewEntry()
    {
        $current_model        = Model::factory()->create(['type' => 'freelancer']);
        $user                 = User::factory()->make()->toArray();
        $user['password']     = '123456';
        $driver               = Driver::factory()->make()->toArray();
        $truck                = Truck::factory()->make()->toArray();
        $current_driver       = $current_model->driver()->create($driver);
        $current_truck        = $current_driver->truck()->create($truck);
        $this->data           = array_merge($this->data, $user);
        $this->data['driver'] = $current_driver->toArray();
        $this->data['truck']  = $current_truck->toArray();
        $current_model->user()->create($user);

        return $current_model;
    }

    /**
     * get the id
     *
     * @return int
     */
    protected function getId()
    {
        return $this->getNewEntry()->id;
    }

    /**
     * Display a listing of the resource.
     *
     * @return void
     */
    public function testItShouldGetListingOfTheResource()
    {
        $this->getNewEntry();
        $current_json              = $this->json;
        $current_json['data']      = [];
        $current_json['data']['*'] = $this->json['data'];

        $this->json('GET', $this->base_url, [], $this->getHeaders())
             ->assertStatus(200)
             ->assertJsonStructure($current_json);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @return void
     */
    public function testItShouldStoreNewlyCreatedResource()
    {
        $user = User::factory()->make()->toArray();
        $this->data = array_merge($this->data, $user);
        $this->data['password'] = '123456';

        if($this->data['type'] == 'freelancer'){
            $this->data['driver'] = Driver::factory()->make()->toArray();
            $this->data['truck']  = Truck::factory()->make()->toArray();
        }

        $this->json('POST', $this->base_url, $this->data, $this->getHeaders())
             ->assertStatus(201)
             ->assertJsonStructure($this->json);
    }

    /**
     * Display the specified resource.
     *
     * @return void
     */
    public function testItShouldGetSpecifiedResource()
    {
        $current_json           = $this->json;
        $current_json['data'][] = 'driver';
        $current_json['data'][] = 'user';
        
        $this->json('GET', $this->base_url . $this->getId(), [], $this->getHeaders())
             ->assertStatus(200)
             ->assertJsonStructure($current_json);
    }

    /**
     * update a resource in storage.
     *
     * @return void
     */
    public function testItShouldUpdateSpecifiedResource()
    {
        $id = $this->getId();
        $this->data['name']                             = 'updated name';
        $this->data['driver']['driving_license_number'] = 'A1UB89AFPO';
        $this->data['truck']['plate_number']            = 'YUV987';

        $this->json('PUT', $this->base_url . $id, $this->data, $this->getHeaders())
             ->assertStatus(200)
             ->assertJsonStructure($this->json);
    }

    /**
     * Remove the specified resource from storage.
     *
     * @return void
     */
    public function testItShouldRemoveSpecifiedResource()
    {
        $this->json['data'] = [];
        $this->json('DELETE', $this->base_url . $this->getId(), [], $this->getHeaders())
             ->assertStatus(200)
             ->assertJsonStructure($this->json);
    }
}
