<?php

namespace Core\Provider\Models;

use Core\Base\Models\Base;
use Illuminate\Database\Eloquent\Casts\Attribute;

class Provider extends Base
{
    /**
     * The "booted" method of the model.
     *
     * @return void
     */
    protected static function booted()
    {
        static::addGlobalScope(new Scopes\TypeScope);
    }

    /**
     * get the user.
     */
    public function user()
    {
        return $this->morphOne(\Core\Auth\Models\User::class, 'userable');
    }

    /**
     * get the contract.
     */
    public function contract()
    {
        return $this->morphOne(\Core\Contract\Models\Contract::class, 'contractable');
    }

    /**
     * get the drivers (in case of partner).
     */
    public function drivers()
    {
        return $this->hasMany(Driver::class, 'provider_id');
    }

    /**
     * get the driver (in case of freelancer).
     */
    public function driver()
    {
        return $this->hasOne(Driver::class, 'provider_id');
    }

    /**
     * get the shipments.
     */
    public function shipments()
    {
        return $this->hasMany(\Core\Shipment\Models\Shipment::class, 'provider_id');
    }

    /**
     * get the interests.
     */
    public function interests()
    {
        return $this->hasMany(\Core\Shipment\Models\Interest::class, 'provider_id');
    }

    /**
     * get the interest (in the shipment details).
     */
    public function interest()
    {
        return $this->hasOne(\Core\Shipment\Models\Interest::class, 'provider_id')->where('shipment_id', request()->shipment);
    }

    /**
     * check if the provider made an interest on a shipment or not.
     */
    public function interested($shipment_id)
    {
        return $this->interests()->where('shipment_id', $shipment_id)->first();
    }

    /**
     * get the backhauls.
     */
    public function backhauls()
    {
        return $this->hasMany(\Core\Shipment\Models\Backhaul::class, 'provider_id');
    }

    /**
     * Interact with the provider's price.
     *
     * @return \Illuminate\Database\Eloquent\Casts\Attribute
     */
    public function price(): Attribute
    {
        return new Attribute(
            get: fn ($value) => $value,
            set: fn ($value) => $value
        );
    }

    /**
     * Interact with the provider's margin.
     *
     * @return \Illuminate\Database\Eloquent\Casts\Attribute
     */
    public function margin(): Attribute
    {
        return new Attribute(
            get: fn ($value) => $value
        );
    }

    /**
     * Interact with the provider's margin percentage.
     *
     * @return \Illuminate\Database\Eloquent\Casts\Attribute
     */
    public function marginPercentage(): Attribute
    {
        return new Attribute(
            get: fn ($value) => $value
        );
    }

    /**
     * Interact with the provider's is backhauling.
     *
     * @return \Illuminate\Database\Eloquent\Casts\Attribute
     */
    public function isBackhauling(): Attribute
    {
        return new Attribute(
            set: fn ($shipment_id) => $this->backhauls()->where('shipment_id', $shipment_id)
                                                        ->where('allow_processing', true)
                                                        ->first() ? true : false
        );
    }

    /**
     * get the requests.
     */
    public function requests()
    {
        return $this->morphMany(\Core\Financial\Models\Request::class, 'beneficiaryable');
    }

    /**
     * get the eligible partner drivers.
     */
    public function eligibleDrivers($truck_type_id)
    {
        $trucks = Truck::where('truck_type_id', $truck_type_id)
                       ->whereIn('driver_id', $this->drivers->pluck('id'))
                       ->get();

        return $this->drivers()->whereIn('id', $trucks->pluck('driver_id'))->get();
    }

    /**
     * get the company.
     */
    public function company()
    {
        return $this->morphOne(\Core\Common\Models\Company::class, 'companyable');
    }
}
