<?php

use Core\Financial\Models\Wallet;
use Core\Financial\Models\Promotion;
use Core\Financial\Models\Scopes\StatusScope;
use Core\Auth\Models\User;
use Core\Shipment\Models\Shipment;
use Carbon\Carbon;

if (!function_exists('handle_wallet_transaction')) {
    /**
     * handle wallet transaction.
     *
     * @param  Wallet $wallet
     * @param  array  $transaction
     * @return void
     */
    function handle_wallet_transaction(Wallet $wallet, $transaction)
    {
        $bank_transfer_value = config('core_financial.payment.methods.bank-transfer.value');
        $user                = $wallet->user;

        if ($transaction['transactionable_type'] == $bank_transfer_value && $user->account) {
            $account = $user->account;
            $payload = [
                'trackid'           => time() . "-$bank_transfer_value", // prefix here is just for testing and will be removed when I'm done from testing
                'customerEmail'     => $user->email,
                'customerIp'        => request()->ip(),
                'merchantIp'        => request()->ip(),
                'country'           => 'SA', // needs to be dynamic
                'currency'          => $wallet->currency,
                'amount'            => $transaction['amount'],
                'paymentCycle'      => 'RT',
                'instrumentType'    => 'Default',
                'subInterfaceCode'  => '',
                'b2bCustomerName'   => $user->userable->name,
                'b2bPaymentDetails' => [
                    [
                        'beneficiaryBank'             => $account->bank,
                        'beneficiaryBankAddress'      => $account->bank_address,
                        'beneficiaryBankClearingCode' => $account->bank_clearing_code,
                        'beneficiaryBankCode'         => $account->bank_code,
                        'beneficiaryAccountNo'        => $account->account_number,
                        'beneficiaryName'             => $account->name,
                        'beneficiaryAddress'          => $account->address,
                        'paymentDetails'              => time(), // the length has to be max: 12
                        'beneficiaryCurrency'         => $account->currency,
                        'beneficiaryAmount'           => $transaction['amount']
                    ]
                ],
            ];

            $urway    = new \Core\Financial\PaymentGateways\Urway($payload);
            $transfer = $urway->transfer();

            if ($transfer['responseCode'] == '000') {
                $transaction['amount'] = - $transaction['amount'];
                $wallet->transactions()->create($transaction);
            }
        } else if ($transaction['transactionable_type'] != $bank_transfer_value) {
            $wallet->transactions()->create($transaction);
        }
    }
}

if (!function_exists('handle_promotions_trackers')) {
    /**
     * handle promotions trackers.
     *
     * @param  User $user
     * @param  Shipment $shipment
     * @return void
     */
    function handle_promotions_trackers(User $user, Shipment $shipment = null)
    {
        $data = [
            'current_time'   => Carbon::now(),
            'from_city_ids'  => [],
            'to_city_ids'    => [],
            'truck_type_ids' => [],
        ];
        $user_promotions = $user->promotionsTrackers()
                                ->withoutGlobalScope(StatusScope::class)
                                ->where('status', config('core_financial.statuses.promotion.in-progress.value'))
                                ->get();
        $current_promotions = $user_promotions ? $user_promotions->pluck('promotion_id') : [];
        $all_promotions     = Promotion::where('status', config('core_financial.statuses.promotion.available.value'))
                                       ->whereNotIn('id', $current_promotions)
                                       ->get();

        if ($user->userable && $user->userable->contract) {
            $items = $user->userable->contract->items;

            foreach ($items as $item) {
                $data['from_city_ids'][]  = $item->from_city_id;
                $data['to_city_ids'][]    = $item->to_city_id;
                $data['truck_type_ids'][] = $item->truck_type_id;
            }
        }

        if ($shipment) {
            $data['shipment'] = $shipment;

            foreach ($user_promotions as $promotion) {
                $current_details = promotion_eligibility($promotion->details, $data);
    
                if ($current_details) {
                    $status         = $promotion->status;
                    $expired_status = config('core_financial.statuses.promotion.expired.value');

                    if (isset($current_details['status']) && $current_details['status']) {
                        $status = $current_details['status'];
                    }

                    if ($current_details['conditions']['achievement'] == 'all' &&
                        isset($current_details['conditions']['end_time']['status']) &&
                        $current_details['conditions']['end_time']['status'] == $expired_status) {
                        $status = config('core_financial.statuses.promotion.failed.value');
                    }

                    $promotion->update([
                        'status'  => $status,
                        'details' => $current_details
                    ]);
                }
            }
        }

        foreach ($all_promotions as $promotion) {
            $current_details = promotion_eligibility($promotion->details, $data);
    
            if ($current_details) {
                $promotion->trackers()->create([
                    'user_id' => $user->id,
                    'status'  => config('core_financial.statuses.promotion.in-progress.value'),
                    'details' => $current_details
                ]);
            }
        }
    }
}

if (!function_exists('promotion_eligibility')) {
    /**
     * promotion eligibility.
     *
     * @param  array   $details
     * @param  array   $data
     * @return array
     */
    function promotion_eligibility($details, $data)
    {
        $result          = [];
        $current_details = $details;
        $conditions      = $details['conditions'];
        $actions         = $details['actions'];

        if (!isset($current_details['conditions']['delivered_shipments']['current_value'])) {
            $current_details['conditions']['delivered_shipments']['current_value'] = 0;
            $current_details['conditions']['delivered_shipments']['ids'] = [];
            $current_details['actions']['total']['current_value'] = $actions['total']['type'] == 'fixed' ?
                                                                    $actions['total']['value'] : 0.00;
        }

        if (isset($conditions['start_time'])) {
            $result[] = $data['current_time']->gte(Carbon::parse($conditions['start_time']['value'])) ?? false;
        }

        if (isset($conditions['end_time'])) {
            if ($data['current_time']->lte(Carbon::parse($conditions['end_time']['value']))) {
                $result[] = true;
            } else {
                $result[] = false;
                $current_details['conditions']['end_time']['status'] = config('core_financial.statuses.promotion.expired.value');
            }
        }

        if (isset($conditions['from_city_id'])) {
            $result[] = in_array($conditions['from_city_id']['value'], $data['from_city_ids']) ?? false;
        }

        if (isset($conditions['to_city_id'])) {
            $result[] = in_array($conditions['to_city_id']['value'], $data['to_city_ids']) ?? false;
        }

        if (isset($conditions['truck_type_id'])) {
            $result[] = in_array($conditions['truck_type_id']['value'], $data['truck_type_ids']) ?? false;
        }

        if (($conditions['achievement'] == 'all' && !in_array(false, $result)) ||
            ($conditions['achievement'] == 'any' && in_array(true, $result))) {

            if (isset($data['shipment'])) {
                $current_details['conditions']['delivered_shipments']['current_value'] += 1;
                $current_details['conditions']['delivered_shipments']['ids'][] = ['id' => $data['shipment']['id']];

                if ($current_details['conditions']['delivered_shipments']['current_value'] == 
                    $current_details['conditions']['delivered_shipments']['value']) {
                    $current_details['status'] = config('core_financial.statuses.promotion.achieved.value');

                    if ($actions['total']['type'] == 'percentage') {
                        $current_total_value = 0.00;

                        foreach ($current_details['conditions']['delivered_shipments']['ids'] as $id_object) {
                            $shipment = Shipment::find($id_object['id']);
                            $current_total_value += $shipment->cost;
                        }
    
                        $actions['total']['current_value'] = ($actions['total']['value'] * $current_total_value) / 100;
                        $current_details['actions']['total']['current_value'] = $actions['total']['current_value'];
                    }

                }
            }

            return $current_details;
        }

        return [];
    }
}