<?php

namespace Core\Financial\Jobs;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldBeUnique;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Core\Contract\Models\Contract;
use Core\Financial\Models\Invoice;
use Illuminate\Support\Facades\Notification;
use Core\Financial\Notifications\CombinedInvoice;

class RequestCollection implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    /**
     * contract
     * 
     * @var Contract
     */
    protected $contract;

    /**
     * Create a new job instance.
     *
     * @param  Contract $contract
     * @return void
     */
    public function __construct(Contract $contract)
    {
        $this->contract = $contract;
    }

    /**
     * Execute the job.
     *
     * @return void
     */
    public function handle()
    {
        $total = 0.00;
        $invoice_items = [];
        $provider = $this->contract->contractable;
        $user     = $provider->user;
        $wallet   = $user->wallet;
        $requests = $provider->requests()->where('status', config('core_financial.statuses.request.pending.value'))
                                         ->get();

        foreach ($requests as $request) {
            $invoice_items[] = $request->invoiceItem()->make(['price' => $request->total])->toArray();
            $total += $request->total;
            $request->update(['status' => config('core_financial.statuses.request.paid.value')]);
        }

        if ($total > 0) {
            $invoice = Invoice::create([
                'invoiceable_type' => 'request',
                'status'           => config('core_financial.statuses.invoice.paid.value'),
                'total'            => $total,
                'due_date'         => date('Y-m-d'),
                'currency'         => 'SAR' // needs to be dynamic
            ]);

            $invoice->items()->createMany($invoice_items);

            handle_wallet_transaction($wallet, $invoice->walletTransaction()->make(['amount' => $total])->toArray());
            handle_wallet_transaction($wallet, [
                'transactionable_type' => config('core_financial.payment.methods.bank-transfer.value'),
                'amount'               => number_format($wallet->balance + $total, 2, '.', '') // Adding here because the balance is not updated
            ]);

            $admins = \Core\Auth\Models\User::where('userable_type', null)->get();
            $users  = $admins->push($user);
            
            Notification::send($users, new CombinedInvoice($invoice));
        }

        self::dispatch($this->contract)->delay(now()->addDays($this->contract->payment_terms));
    }
}
