<?php

namespace Core\Financial\Jobs;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldBeUnique;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Core\Contract\Models\Contract;
use Core\Financial\Models\Invoice;
use Illuminate\Support\Facades\Notification;
use Core\Financial\Notifications\CombinedInvoice;

class InvoiceCollection implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    /**
     * contract
     * 
     * @var Contract
     */
    protected $contract;

    /**
     * Create a new job instance.
     *
     * @param  Contract $contract
     * @return void
     */
    public function __construct(Contract $contract)
    {
        $this->contract = $contract;
    }

    /**
     * Execute the job.
     *
     * @return void
     */
    public function handle()
    {
        $total    = 0.00;
        $invoices = [];
        $shipper  = $this->contract->contractable;
        $orders   = $shipper->orders()->where('status', '!=', config('core_shipment.statuses.order.paid.value'))
                                      ->where('pickup_date', '<=', date('Y-m-d'))
                                      ->get();

        foreach ($orders as $order) {
            $current_invoice = $order->invoice;
            if ($current_invoice->status == config('core_financial.statuses.invoice.pending.value')) {
                $invoices[] = $current_invoice->item()->make(['price' => $current_invoice->total])->toArray();
                $total += $current_invoice->total;
                $current_invoice->update(['status' => config('core_financial.statuses.invoice.paid.value')]);
            }
        }

        if ($total > 0) {
            $combined_invoice = Invoice::create([
                'invoiceable_type' => 'invoice',
                'status'           => config('core_financial.statuses.invoice.paid.value'),
                'total'            => $total,
                'due_date'         => date('Y-m-d'),
                'currency'         => 'SAR' // needs to be dynamic
            ]);

            $combined_invoice->items()->createMany($invoices);

            $admins = \Core\Auth\Models\User::where('userable_type', null)->get();
            $users  = $admins->push($shipper->user);
            
            Notification::send($users, new CombinedInvoice($combined_invoice));
        }

        self::dispatch($this->contract)->delay(now()->addDays($this->contract->payment_terms));
    }
}
