<?php

namespace Core\Base\Controllers\API;

use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Database\Eloquent\Model;
use Core\Base\Traits\Response\SendResponse;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class Controller extends \App\Http\Controllers\Controller
{
    use SendResponse;

    /**
     * the request
     *
     * @var FormRequest
     */
    protected $request;

    /**
     * the eloquent model
     *
     * @var Model
     */
    protected $model;

    /**
     * the eloquent API resource
     *
     * @var string
     */
    protected $resource;

    /**
     * Init.
     *
     * @codeCoverageIgnore
     * @param FormRequest $request
     * @param Model       $model
     * @param string      $resource
     * @return void
     */
    public function __construct(FormRequest $request, Model $model, $resource)
    {
        $this->request  = $request;
        $this->model    = $model;
        $this->resource = $resource;

        Log::info('API Request', ['request' => $this->request->all()]);
    }

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function index()
    {
        return $this->sendResponse($this->resource::collection(
            $this->model->paginate($this->request->paginate ? $this->request->paginate : 30)
        )->response()->getData(true));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function store()
    {
        return $this->sendResponse(
            new $this->resource($this->dbTransaction($this->model)),
            'successfully created.',
            true,
            201
        );
    }

    /**
     * Display the specified resource.
     *
     * @param  int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function show($id)
    {
        return $this->sendResponse(new $this->resource($this->model->find($id)));
    }

    /**
     * Update a resource in storage.
     *
     * @param  int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function update($id)
    {
        return $this->sendResponse(
            new $this->resource($this->dbTransaction($this->model->find($id), 'update')),
            'successfully updated.'
        );
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int $id
     * @return \Illuminate\Http\JsonResponse
     * @throws \Exception
     */
    public function destroy($id)
    {
        $this->dbTransaction($this->model->find($id), 'delete');
        return $this->sendResponse([], 'successfully deleted.');
    }

    /**
     * Create DB Transaction.
     *
     * @param  Model $current_model
     * @param  string $operation
     * @return Model
     * @throws \Exception
     */
    public function dbTransaction(Model $current_model, $operation = 'create')
    {
        $result_model = $current_model;
        $all_request  = $this->request->all();

        DB::transaction(function () use ($current_model, $operation, &$result_model, $all_request) {
            $result = $operation == 'delete' ? $current_model->delete() : $current_model->{$operation}($all_request);

            if($operation == 'create') {
                $result_model = $result;
            }

        }, 3);

        return $result_model;
    }
}
