<?php

namespace Core\Auth\Tests\Feature;

use Core\Base\Tests\TestCase;
use Core\Auth\Models\User as Model;

class AuthTest extends TestCase
{
    /**
     * the base url
     *
     * @var string
     */
    protected $base_url;

    /**
     * the data that will be sent in the request (create/update)
     *
     * @var array
     */
    protected $data;

    /**
     * the json response
     *
     * @var array
     */
    protected $json;

    /**
     * setting up
     *
     * @throws \ReflectionException
     * @return void
     */
    public function setUp(): void
    {
        parent::setUp();
        \Artisan::call('migrate',['-vvv' => true]);
        \Artisan::call('passport:install',['-vvv' => true]);

        $this->base_url     = $this->getApiBaseUrl() . 'auth/';
        $this->data         = Model::factory()->make()->toArray();
        $this->json         = $this->getJsonStructure();
        $this->json['data'] = ['id'];

        foreach ($this->data as $key => $value) {
            $this->json['data'][] = $key;
        }
    }

    /**
     * successfully login the user.
     *
     * @return void
     */
    public function testItShouldLoginTheUserSuccessfully()
    {
        \Artisan::call('migrate',['-vvv' => true]);
        \Artisan::call('passport:install',['-vvv' => true]);

        $body = Model::factory()->create()->toArray();
        // we are adding the password again as it is hidden for serialization (checkout the user model)
        $body['password']  = '123456';
        $body['fcm_token'] = 'dummy-fcm-token';
        
        $response = $this->getJsonStructure();
        $response['data'] = [
            'token' => ['access', 'type', 'expires_at'],
            'user'
        ];

        $this->json('POST', $this->base_url . 'login', $body, $this->getHeaders(false))
             ->assertStatus(200)
             ->assertJsonStructure($response);
    }

    /**
     * invalid login.
     *
     * @return void
     */
    public function testItShouldBeInvalidLogin()
    {
        // we are adding this var as the password is hidden for serialization (checkout the user model)
        $body = array_merge(['password' => 'wrong-password-here'], $this->data);
        $body['fcm_token'] = 'dummy-fcm-token';

        $this->json('POST', $this->base_url . 'login', $body, $this->getHeaders(false))
             ->assertStatus(401);
    }

    /**
     * Display the auth user resource.
     *
     * @return void
     */
    public function testItShouldGetAuthUserResource()
    {
        $this->json('GET', $this->base_url . 'me', [], $this->getHeaders())
             ->assertStatus(200)
             ->assertJsonStructure($this->json);
    }

    /**
     * update the auth user resource in storage.
     *
     * @return void
     */
    public function testItShouldUpdateAuthUserResource()
    {
        $this->json('PUT', $this->base_url . 'update', $this->data, $this->getHeaders())
             ->assertStatus(200)
             ->assertJsonStructure($this->json);
    }
}
