<?php

namespace Core\Auth\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Laravel\Passport\HasApiTokens;
use Core\Auth\Database\Factories\UserFactory;
use Illuminate\Database\Eloquent\Casts\Attribute;
use OwenIt\Auditing\Contracts\Auditable;
use Core\Base\Traits\Model\File;

class User extends Authenticatable implements Auditable
{
    use HasApiTokens, HasFactory, Notifiable, \OwenIt\Auditing\Auditable, File;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'userable_type',
        'userable_id',
        'name',
        'email',
        'phone',
        'national_id',
        'avatar',
        'password',
        'fcm_tokens'
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var array<int, string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'email_verified_at' => 'datetime',
    ];
 
    /**
     * Create a new factory instance for the model.
     *
     * @return \Illuminate\Database\Eloquent\Factories\Factory
     */
    protected static function newFactory()
    {
        return UserFactory::new();
    }

    /**
     * Interact with the user's password.
     *
     * @return \Illuminate\Database\Eloquent\Casts\Attribute
     */
    public function password(): Attribute
    {
        return new Attribute(
            set: fn ($value) => bcrypt($value),
        );
    }

    /**
     * Interact with the user's fcm tokens.
     *
     * @return \Illuminate\Database\Eloquent\Casts\Attribute
     */
    public function fcmTokens(): Attribute
    {
        return new Attribute(
            get: fn ($value) => $value ? explode(',', $value) : []
        );
    }

    /**
     * set the fcm token in the user's fcm tokens attribute.
     */
    public function setFcmTokenAttribute($value)
    {
        $tokens = $this->fcm_tokens;

        if(!in_array($value, $tokens)) {
            $tokens[] = $value;

            $this->attributes['fcm_tokens'] = implode(',', $tokens);
        }
    }

    /**
     * Get the parent userable model.
     */
    public function userable()
    {
        return $this->morphTo();
    }

    /**
     * Specifies the user's FCM tokens
     *
     * @return string|array
     */
    public function routeNotificationForFcm()
    {
        return $this->fcm_tokens;
    }

    /**
     * get the wallet
     * (as per the business needs, we only need 1 wallet but maybe in the future we can have more than 1 wallet).
     * 
     */
    public function wallet()
    {
        return $this->hasOne(\Core\Financial\Models\Wallet::class, 'user_id');
    }

    /**
     * get the bank account
     * (as per the business needs, we only need 1 account but maybe in the future we can have more than 1 account).
     * 
     */
    public function account()
    {
        return $this->hasOne(\Core\Financial\Models\Account::class, 'user_id');
    }

    /**
     * get the promotions trackers
     * 
     */
    public function promotionsTrackers()
    {
        return $this->hasMany(\Core\Financial\Models\PromotionTracker::class, 'user_id');
    }

    /**
     * Interact with the avatar.
     *
     * @return \Illuminate\Database\Eloquent\Casts\Attribute
     */
    public function avatar(): Attribute
    {
        return new Attribute(
            get: fn ($value) => $this->getFilePath($value),
            set: fn ($value) => $this->setFilePath($value, 'avatar')
        );
    }
}
