<?php

namespace Core\Auth\Controllers\API\V1;

use Core\Auth\Requests\UserRequest;
use Core\Auth\Requests\AuthRequest as CustomRequest;
use Core\Auth\Models\User as Model;
use Core\Auth\Resources\UserResource as Resource;
use \Illuminate\Support\Facades\Hash;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Password;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Str;
use Carbon\Carbon;

class AuthController extends \Core\Base\Controllers\API\Controller
{
    /**
     * Create a new AuthController instance.
     *
     * @return void
     */
    public function __construct()
    {

    }

    /**
     * Get a JWT via given credentials.
     *
     * @param CustomRequest $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function login(CustomRequest $request)
    {
        $user = Model::where('email', $request->email)->first();

        if ($user && Hash::check($request->password, $user->password)) {

            if($request->fcm_token) {
                $user->fcm_token = $request->fcm_token;
                $user->update();
            }

            // if ($user->userable_type && !$user->userable->is_active) {
            //     return $this->sendResponse([], 'your account is registered but needs to be activated', false, 401);
            // }

            return $this->respondWithToken($user);
        }

        return $this->sendResponse([], 'please login with the correct data', false, 401);
    }

    /**
     * Get the authenticated User.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function me()
    {
        return $this->sendResponse(new Resource(auth()->user()));
    }

    /**
     * update the data of the authenticated user.
     *
     * @param UserRequest $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function updateMe(UserRequest $request)
    {
        auth()->user()->update($request->all());
        return $this->sendResponse(new Resource(auth()->user()), 'successfully updated.');
    }

    /**
     * Log the user out (Invalidate the token).
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function logout()
    {
        auth()->logout();
        return $this->sendResponse([], 'Successfully logged out.');
    }

    /**
     * Get the token array structure.
     *
     * @param  Model $user
     * @return \Illuminate\Http\JsonResponse
     */
    protected function respondWithToken(Model $user)
    {
        $token = $user->createToken('App Token')->accessToken;
        
        return $this->sendResponse([
            'token' => [
                'access'     => $token,
                'type'       => 'Bearer',
                'expires_at' => $user->tokens()->first()->expires_at,
            ],
            'user' => new Resource($user)
        ]);
    }
    
    public function sendResetLinkEmail(Request $request)
    {
        $request->validate([
            'email' => 'required|email',
        ]);

        $user = DB::table('users')->where('email', $request->email)->first();
        if (!$user) {
            return $this->sendResponse([], 'No user found with this email', false, 404);
        }

        $token = Str::random(6);

        DB::table('password_resets')->updateOrInsert(
            ['email' => $request->email],
            [
                'email' => $request->email,
                'token' => $token,
                'created_at' => Carbon::now()
            ]
        );

        // $resetLink = url('/api/password/reset/' . $token . '?email=' . urlencode($request->email));
        $resetLink = $token;

        $this->sendResetEmail($request->email, $resetLink);

        return $this->sendResponse([], 'Password reset link sent.');
    }

    protected function sendResetEmail($email, $resetLink)
    {
        $subject = 'Reset Password OTP';
        $message = "Hello,\n\nYou requested a password reset. Copy the code below to reset your password:\n\n";
        $message .= $resetLink . "\n\n";
        $message .= "If you did not request a password reset, no further action is required.\n\n";
        $message .= "Thanks,\nEload team";

        Mail::raw($message, function ($mail) use ($email, $subject) {
            $mail->to($email)
                 ->subject($subject);
        });
    }
    
    public function resetPassword(Request $request)
    {
        $request->validate([
            'token' => 'required',
            'email' => 'required|email',
            'password' => 'required|string|min:8|confirmed',
        ]);

        $passwordReset = DB::table('password_resets')
            ->where([
                ['email', $request->email],
                ['token', $request->token],
            ])->first();

        if (!$passwordReset) {
            return $this->sendResponse([], 'Invalid token or email.', false, 400);
        }

        if (Carbon::parse($passwordReset->created_at)->addMinutes(120)->isPast()) {
            return $this->sendResponse([], 'This password reset token is expired.', false, 400);
        }

        DB::table('users')->where('email', $request->email)->update([
            'password' => Hash::make($request->password),
            'remember_token' => Str::random(60),
        ]);

        DB::table('password_resets')->where('email', $request->email)->delete();
            return $this->sendResponse([], 'Password has been successfully reset.');
    }
}